<?php
/**
 * Plugin Name:       bbPress Discussion Forum Schema & SEO Rich Snippets
 * Plugin URI:        https://sitetinker.com/bbpress-discussion-forum-schema-seo-rich-snippets-wordpress-plugin/
 * Description:       Automatically adds advanced "DiscussionForumPosting" structured data to bbPress topics to enable Forum Rich Results in Google. Prioritizes forum data over standard article schemas for better SEO visibility.
 * Version:           1.0
 * Author:            SiteTinker
 * Author URI:        https://SiteTinker.com/
 * License:           GPL v2 or later
 * Text Domain:       bbp-forum-schema
 */

if ( ! defined( 'ABSPATH' ) ) exit;

// =============================================================================
// PART 1: THE PREVENTER
// Tell SEO plugins to stop outputting Schema on Topic pages.
// =============================================================================
add_action('wp', 'bbp_schema_disable_conflicts', 99);

function bbp_schema_disable_conflicts() {
    // Only run on single bbPress topics
    if ( function_exists('bbp_is_single_topic') && bbp_is_single_topic() ) {
        add_filter( 'wpseo_json_ld_output', '__return_false' );       // Yoast
        add_filter( 'rank_math/json_ld', '__return_false' );          // RankMath
        add_filter( 'aioseo_schema_disable', '__return_true' );       // AIOSEO
        add_filter( 'seopress_json_ld_enable', '__return_false' );    // SEOPress
        add_filter( 'sq_json_ld_enable', '__return_false' );          // Squirrly
        add_filter( 'generate_schema_type', '__return_false' );       // GeneratePress
    }
}

// =============================================================================
// PART 2: THE SCRUBBER (Output Buffering)
// Intercepts HTML to remove hardcoded Theme Microdata and "Article" scripts.
// =============================================================================
add_action( 'template_redirect', 'bbp_schema_buffer_start', 1 );

function bbp_schema_buffer_start() {
    if ( function_exists('bbp_is_single_topic') && bbp_is_single_topic() ) {
        // Start buffering output
        ob_start( 'bbp_schema_scrubber_callback' );
    }
}

function bbp_schema_scrubber_callback( $buffer ) {
    
    // 1. Remove HTML Microdata attributes (itemtype="...Article")
    // Catches http, https, and cases without protocol (schema.org/Article)
    // Catches Article, BlogPosting, NewsArticle, WebPage, CreativeWork, SocialMediaPosting
    $buffer = preg_replace( 
        '/itemtype\s*=\s*["\'](https?:\/\/)?schema\.org\/(Article|BlogPosting|NewsArticle|WebPage|CreativeWork|SocialMediaPosting)["\']/i', 
        '', 
        $buffer 
    );
    
    // 2. Remove "itemscope" if left dangling
    $buffer = str_replace( array(' itemscope="itemscope"', ' itemscope=""', ' itemscope'), '', $buffer );

    // 3. Remove "hentry" class (prevents phantom Microformats)
    $buffer = preg_replace_callback( '/class=["\'](.*?)["\']/', function( $matches ) {
        // Strips "hentry", "h-entry", "type-post"
        $cleaned = preg_replace( '/\b(hentry|h-entry|type-post|post-[0-9]+)\b/', '', $matches[1] );
        return 'class="' . trim( $cleaned ) . '"';
    }, $buffer );

    // 4. Remove JSON-LD Scripts that contain "Article" schema (The Nuclear Option)
    $buffer = preg_replace_callback( '#<script[^>]*application/ld\+json[^>]*>(.*?)</script>#is', function( $matches ) {
        // If the script declares it is an Article, remove the whole block
        if ( preg_match( '/"@type"\s*:\s*"(Article|BlogPosting|NewsArticle)"/i', $matches[1] ) ) {
            return '<!-- [bbPress Fix] Conflicting Article Schema Removed -->';
        }
        return $matches[0];
    }, $buffer );

    return $buffer;
}

// =============================================================================
// PART 3: THE GENERATOR
// Adds the correct DiscussionForumPosting Schema
// =============================================================================
add_action('wp_head', 'bbp_add_forum_schema_json_ld', 100);

function bbp_add_forum_schema_json_ld() {
    // Safety check
    if ( ! function_exists('bbp_is_single_topic') || ! bbp_is_single_topic() ) {
        return;
    }

    // --- A. GET TOPIC DATA ---
    $topic_id = bbp_get_topic_id();
    $topic_content_raw = bbp_get_topic_content($topic_id);
    $topic_content = wp_strip_all_tags($topic_content_raw);
    
    // Extract Topic Image
    preg_match('/<img.+src=[\'"]([^\'"]+)[\'"].*>/i', $topic_content_raw, $topic_img);

    // --- B. GET REPLIES LOOP ---
    $reply_ids = bbp_get_all_child_ids($topic_id, bbp_get_reply_post_type());
    $comments = [];

    if ( ! empty( $reply_ids ) ) {
        foreach ($reply_ids as $reply_id) {
            $reply_content_raw = bbp_get_reply_content($reply_id);
            $reply_content = wp_strip_all_tags($reply_content_raw);
            
            // Extract Reply Image
            preg_match('/<img.+src=[\'"]([^\'"]+)[\'"].*>/i', $reply_content_raw, $reply_img);

            $comments[] = [
                "@type" => "Comment",
                "author" => [
                    "@type" => "Person",
                    "name" => bbp_get_reply_author_display_name($reply_id),
                    "url" => bbp_get_user_profile_url(bbp_get_reply_author_id($reply_id))
                ],
                "datePublished" => get_the_date('c', $reply_id),
                "url" => bbp_get_reply_url($reply_id),
                "text" => $reply_content, // "text" is preferred over "articleBody" for comments
                "image" => !empty($reply_img[1]) ? esc_url($reply_img[1]) : null
            ];
        }
    }

    // --- C. CONSTRUCT SCHEMA ---
    $topic_schema = [
        "@context" => "https://schema.org",
        "@type" => "DiscussionForumPosting",
        "headline" => bbp_get_topic_title($topic_id),
        "author" => [
            "@type" => "Person",
            "name" => bbp_get_topic_author_display_name($topic_id),
            "url" => bbp_get_user_profile_url(bbp_get_topic_author_id($topic_id))
        ],
        "datePublished" => get_the_date('c', $topic_id),
        "dateModified" => get_the_modified_date('c', $topic_id),
        
        // FIX: Point to the Topic URL (not the last loop variable)
        "url" => bbp_get_topic_permalink($topic_id),
        
        // FIX: Use "text" instead of "articleBody" (Google Forum Standard)
        "text" => $topic_content,
        
        "image" => !empty($topic_img[1]) ? esc_url($topic_img[1]) : null,
        "comment" => $comments,
        
        // ADDED: Interaction Stats (Crucial for Forum snippet)
        "interactionStatistic" => [
            "@type" => "InteractionCounter",
            "interactionType" => "https://schema.org/CommentAction",
            "userInteractionCount" => count($comments)
        ]
    ];

    // --- D. OUTPUT ---
    echo PHP_EOL . '<!-- [bbPress Fix] DiscussionForumPosting Schema -->' . PHP_EOL;
    echo '<script type="application/ld+json">' . PHP_EOL;
    echo wp_json_encode($topic_schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    echo PHP_EOL . '</script>';
}